{*******************************************}
{* Main unit for Trigonometry2 application *}
{*******************************************}

unit trigo;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FileUtil, Forms, Controls, Graphics, Dialogs, Menus, StdCtrls, ExtCtrls, LazUTF8, math, trigo_help;

type
  {*********}
  { TfTrigo }
  {*********}
  TfTrigo = class(TForm)
    mMenu: TMainMenu;
    mFile, mFileExit: TMenuItem;
    mOptions, mOptionsRad, mOptionsPrecision: TMenuItem;
    mOptionsPrecision0, mOptionsPrecision1, mOptionsPrecision2, mOptionsPrecision3: TMenuItem;
    mHelp, mHelpHelp, mHelpAbout: TMenuItem;
    StaticText1: TStaticText;
    Label1, Label6, Label8, Label2, Label3, Label14: TLabel;
    rbSin, rbCos, rbTan, rbCotan, rbSec, rbCosec: TRadioButton;
    laUnit, laA, laB, laN, laK: TLabel;
    laInterval, laThetaMin, laThetaMax, laUThetaMin, laUThetaMax: TLabel;
    edA, edB, edN, edK: TEdit;
    edThetaMin, edThetaMax: TEdit;
    edPSolution, edISolutions: TEdit;
    btPi: TButton;
    btSquareroot: TButton;
    btSolve: TButton;
    procedure FormCreate(Sender: TObject);
    procedure mFileExitClick(Sender: TObject);
    procedure mOptionsRadClick(Sender: TObject);
    procedure mOptionsPrecision0Click(Sender: TObject);
    procedure mOptionsPrecision1Click(Sender: TObject);
    procedure mOptionsPrecision2Click(Sender: TObject);
    procedure mOptionsPrecision3Click(Sender: TObject);
    procedure mHelpHelpClick(Sender: TObject);
    procedure mHelpAboutClick(Sender: TObject);
    procedure rbSinChange(Sender: TObject);
    procedure rbCosChange(Sender: TObject);
    procedure rbTanChange(Sender: TObject);
    procedure rbCotanChange(Sender: TObject);
    procedure rbSecChange(Sender: TObject);
    procedure rbCosecChange(Sender: TObject);
    procedure edAClick(Sender: TObject);
    procedure edBClick(Sender: TObject);
    procedure edNClick(Sender: TObject);
    procedure edKClick(Sender: TObject);
    procedure edThetaMinClick(Sender: TObject);
    procedure edThetaMaxClick(Sender: TObject);
    procedure btPiClick(Sender: TObject);
    procedure btSquarerootClick(Sender: TObject);
    procedure btSolveClick(Sender: TObject);
  private
    iPrecision: Integer;
    rA, rB, rN, rK, rTheta0, rTheta1, rTheta2, rThetaMin, rThetaMax: Real;
    sFunction, sA, sB, sN, sK, sThetaMin, sThetaMax: string;
    bHasSolution: Boolean;
    edEdited: TEdit;
  end;

var
  fTrigo: TfTrigo;

implementation

{$R *.lfm}

{ Format real number (F decimal digits) }

function RFormat(R: Real; F: Integer): string;

var
  R0: Real;
  SR: string;
begin
  SR := '';
  if R = 0 then
    SR := '0'
  else begin
    if F >= 0 then begin
      R0 := Round(R * Power(10, F)) / Power(10, F);
      if Abs(R0) < Power(10, -F) then
        SR := FloatToStrF(R, ffExponent, F, 0)
      else
        SR := FloatToStr(R0);
    end;
  end;
  Result := SR;
end;

{ Transform string containing "π" or "√" symbols to real numbers }

procedure GetRealValue(SValue: string; out RValue: Real; out Mess: string);

var
  P: Integer;
  R: Real;
  Ch: Char;
  Err: Boolean;

begin
  Err := False;
  P := UTF8Pos('π', SValue);
  if P > 0 then begin
    // String contains the symbol "π"
    if P <= 3 then begin
      // Must be at string position 1, 2, or 3
      R := Pi;
      if P > 1 then begin
        // Symbol at position 2, or 3
        Ch := UTF8Copy(SValue, P - 1, 1)[1];
        if P = 2 then begin
          // Symbol at position 2: Allowed formats = -π or iπ
          Ch := UTF8Copy(SValue, P - 1, 1)[1];
          if Ch = '-' then
            R := -R
          else if Ch in ['1'..'9'] then
            R *= StrToInt(Ch)
          else
            Err := True;
        end
        else begin
          // Symbol at position 3: Allowed format = -iπ
          if (UTF8Copy(SValue, P - 2, 1) = '-') and (Ch in ['1'..'9']) then
            R *= -StrToInt(Ch)
          else
            Err := True;
        end;
      end;
      // Now check for fraction
      if not Err then begin
        if (UTF8Length(SValue) = P) or (UTF8Length(SValue) = P + 2) then begin
          // Either "π" is at end of string, or "π" is followed by "/j"
          if UTF8Length(SValue) = P + 2 then begin
            // Two characters after "π": Allowed format = π/j
            Ch := UTF8Copy(SValue, P + 2, 1)[1];
            if (UTF8Copy(SValue, P + 1, 1) = '/') and (Ch in ['1'..'9']) then
              R /= StrToInt(Ch)
            else
              Err := True;
          end;
        end
        else
          Err := True;
      end
      else
        Err := True;
    end
    else
      Err := True;
  end
  else begin
    P := UTF8Pos('√', SValue);
    if P > 0 then begin
      // String contains the symbol "√"
      if UTF8Length(SValue) >= P + 1 then begin
        // Character after symbol must be a number
        Ch := UTF8Copy(SValue, P + 1, 1)[1];
        if Ch in ['1'..'9'] then
          R := Sqrt(StrToInt(Ch))
        else
          Err := True;
      end
      else
        Err := True;
      if not Err then begin
        if P > 1 then begin
          // Symbol at position 2 or 3
          Ch := UTF8Copy(SValue, P - 1, 1)[1];
          if P = 2 then begin
            // Symbol at position 2: Allowed formats are -√m or i√m
            Ch := UTF8Copy(SValue, P - 1, 1)[1];
            if Ch = '-' then
              R := -R
            else if Ch in ['1'..'9'] then
              R *= StrToInt(Ch)
            else
              Err := True;
          end
          else if P = 3 then begin
            // Symbol at position 3: Allowed formats are -i√m or j/√m
            if (UTF8Copy(SValue, P - 2, 1) = '-') and (Ch in ['1'..'9']) then
              R *= -StrToInt(Ch)
            else if (UTF8Copy(SValue, P - 2, 1)[1] in ['1'..'9']) and (Ch = '/') then
              R := StrToFloat(UTF8Copy(SValue, P - 2, 1)[1]) / R
            else
              Err := True;
          end
          else if P = 4 then begin
            // Symbol at position 4: Allowed format is -j/√m
            if (UTF8Copy(SValue, P - 3, 1)[1] = '-') and (UTF8Copy(SValue, P - 2, 1)[1] in ['1'..'9']) and (Ch = '/') then
              R := -StrToFloat(UTF8Copy(SValue, P - 2, 1)[1]) / R
            else
              Err := True;
          end
          else
            Err := True;
        end;
        if not Err then begin
          // Now check for fraction
          if (UTF8Length(SValue) = P + 1) or (UTF8Length(SValue) = P + 3) then begin
            // Either "√m" is at end of string, or "√m" is followed by "/j"
            if UTF8Length(SValue) = P + 3 then begin
              // Three characters after "√": Allowed format = √m/j
              Ch := UTF8Copy(SValue, P + 3, 1)[1];
              if (UTF8Copy(SValue, P + 2, 1) = '/') and (Ch in ['1'..'9']) then
                R /= StrToInt(Ch)
              else
                Err := True;
            end;
          end
          else
            Err := True;
        end
        else
          Err := True;
      end;
    end
    else
      R := StrToFloat(SValue);
  end;
  if Err then
    Mess := 'Unsupported parameter value'
  else
    RValue := R;
end;

{ Get unit multiplier for angle transformation to radians }

function DegRadMult(Rad: Boolean): Real;

// The function returns 1 if actual unit = radians, 2π/360 if actual unit is degrees

var
  Mult: Real;

begin
  if Rad then
    Mult := 1
  else
    Mult := 2 * Pi / 360;
  Result := Mult;
end;

{ Read equation parameters from the form }

procedure ReadParams(out SA, SB, SN, SK, SThetaMin, SThetaMax: string; out A, B, N, K, ThetaMin, ThetaMax: Real; out Mess: string);

begin
  Mess := '';
  // Check for empty fields and invalid parameters
  if fTrigo.edA.Text = '' then begin
    Mess := 'Missing parameter a';
    fTrigo.edA.SetFocus;
  end
  else if fTrigo.edA.Text = '0' then begin                                     // a must not be 0
    Mess := 'Invalid parameter a';
    fTrigo.edA.SetFocus;
  end
  else if fTrigo.edB.Text = '' then begin
    Mess := 'Missing parameter b';
    fTrigo.edB.SetFocus;
  end
  else if fTrigo.edN.Text = '' then begin
    Mess := 'Missing parameter n';
    fTrigo.edN.SetFocus;
  end
  else if fTrigo.edN.Text = '0' then begin                                     // n must not be 0
    Mess := 'Invalid parameter n';
    fTrigo.edN.SetFocus;
  end
  else if fTrigo.edK.Text = '' then begin
    Mess := 'Missing parameter k';
    fTrigo.edK.SetFocus;
  end
  else if fTrigo.edThetaMin.Text = '' then begin
    Mess := 'Missing parameter θ' + SUB_1;
    fTrigo.edThetaMin.SetFocus;
  end
  else if fTrigo.edThetaMax.Text = '' then begin
    Mess := 'Missing parameter θ' + SUB_2;
    fTrigo.edThetaMax.SetFocus;
  end;
  if Mess = '' then begin
    // Get string from edit field; transform "π" or "√" symbol to numbers (if any)
    SA := fTrigo.edA.Text; GetRealValue(SA, A, Mess);
    if Mess <> '' then
      fTrigo.edA.SetFocus
    else begin
      SB := fTrigo.edB.Text; GetRealValue(SB, B, Mess);
      if Mess <> '' then
        fTrigo.edB.SetFocus
      else begin
        SN := fTrigo.edN.Text; GetRealValue(SN, N, Mess);
        if Mess <> '' then
          fTrigo.edN.SetFocus
        else begin
          SK := fTrigo.edK.Text; GetRealValue(SK, K, Mess);
          K *= DegRadMult(fTrigo.mOptionsRad.Checked);                         // k is an angle; transform to radians
          if Mess <> '' then
            fTrigo.edK.SetFocus
          else begin
            SThetaMin := fTrigo.edThetaMin.Text; GetRealValue(SThetaMin, ThetaMin, Mess);
            ThetaMin *= DegRadMult(fTrigo.mOptionsRad.Checked);                // θ1 is an angle; transform to radians
            if Mess <> '' then
              fTrigo.edThetaMin.SetFocus
            else begin
              SThetaMax := fTrigo.edThetaMax.Text; GetRealValue(SThetaMax, ThetaMax, Mess);
              ThetaMax *= DegRadMult(fTrigo.mOptionsRad.Checked);              // θ2 is an angle; transform to radians
              if Mess <> '' then
                fTrigo.edThetaMax.SetFocus
            end;
          end;
          if Mess = '' then begin
            if ThetaMin > ThetaMax then begin                                  // θ1 must be smaller than θ2
              Mess := 'Invalid interval values';
              fTrigo.edThetaMax.SetFocus;
            end;
          end;
        end;
      end;
    end;
  end;
end;

{ The following functions return the principal angle and the two angles that are solution of a·f(θ) = b }
{ ----------------------------------------------------------------------------------------------------- }

{ Angles for equation a·sin(θ) = b }

procedure GetAngles_EqSin(A, B: Real; out Theta0, Theta1, Theta2: Real; out HasSolution: Boolean);

var
  ThetaAcute: Real;

begin
  try
    Theta0 := ArcSin(B / A);
    ThetaAcute := ArcSin(Abs(B / A));
    if B / A > 0 then begin
      Theta1 := ThetaAcute;
      Theta2 := Pi - ThetaAcute;
    end
    else begin
      Theta1 := Pi + ThetaAcute;
      Theta2 := 2 * Pi - ThetaAcute;
    end;
    HasSolution := True;
  except
    HasSolution := False;
  end;
end;

{ Angles for equation a·cos(θ) = b }

procedure GetAngles_EqCos(A, B: Real; out Theta0, Theta1, Theta2: Real; out HasSolution: Boolean);

var
  ThetaAcute: Real;

begin
  try
    Theta0 := ArcCos(B / A);
    ThetaAcute := ArcCos(Abs(B / A));
    if B / A > 0 then begin
      Theta1 := ThetaAcute;
      Theta2 := 2 * Pi - ThetaAcute;
    end
    else begin
      Theta1 := Pi - ThetaAcute;
      Theta2 := Pi + ThetaAcute;
    end;
    HasSolution := True;
  except
    HasSolution := False;
  end;
end;

{ Angles for equation a·tan(θ) = b }

procedure GetAngles_EqTan(A, B: Real; out Theta0, Theta1, Theta2: Real; out HasSolution: Boolean);

var
  ThetaAcute: Real;

begin
  try
    Theta0 := ArcTan(B / A);
    ThetaAcute := ArcTan(Abs(B / A));
    if B / A > 0 then begin
      Theta1 := ThetaAcute;
      Theta2 := Pi + ThetaAcute;
    end
    else begin
      Theta1 := Pi - ThetaAcute;
      Theta2 := 2 * Pi - ThetaAcute;
    end;
    HasSolution := True;
  except
    HasSolution := False;
  end;
end;

{ Angles for equation a·cotan(θ) = b }

procedure GetAngles_EqCotan(A, B: Real; out Theta0, Theta1, Theta2: Real; out HasSolution: Boolean);

begin
  if B = 0 then
    HasSolution := False
  else
    GetAngles_EqTan(B, A, Theta0, Theta1, Theta2, HasSolution);
end;

{ Angles for equation a·sec(θ) = b }

procedure GetAngles_EqSec(A, B: Real; out Theta0, Theta1, Theta2: Real; out HasSolution: Boolean);

begin
  if B = 0 then
    HasSolution := False
  else
    GetAngles_EqCos(B, A, Theta0, Theta1, Theta2, HasSolution);
end;

{ Angles for equation a·cosec(θ) = b }

procedure GetAngles_EqCosec(A, B: Real; out Theta0, Theta1, Theta2: Real; out HasSolution: Boolean);

begin
  if B = 0 then
    HasSolution := False
  else
    GetAngles_EqSin(B, A, Theta0, Theta1, Theta2, HasSolution);
end;

{*********}
{ TfTrigo }
{*********}

{ Application start: Initialization }

procedure TfTrigo.FormCreate(Sender: TObject);

begin
  laInterval.Caption := StringReplace(laInterval.Caption, '1', SUB_1, []);
  laInterval.Caption := StringReplace(laInterval.Caption, '2', SUB_2, []);
  laThetaMin.Caption := StringReplace(laThetaMin.Caption, '1', SUB_1, []);
  laThetaMax.Caption := StringReplace(laThetaMax.Caption, '2', SUB_2, []);
  sFunction := 'sin'; iPrecision := 3;
  edEdited := edA;
end;

{ Menu item "File > Exit": Exit application }

procedure TfTrigo.mFileExitClick(Sender: TObject);

begin
  Close;
end;

{ Menu item "Options > Angles in radians": Toggle angle unit degrees/radians }

procedure TfTrigo.mOptionsRadClick(Sender: TObject);

begin
  if mOptionsRad.Checked then begin
    mOptionsRad.Checked := False;
    laUThetaMin.Visible := True;
    laUThetaMax.Visible := True;
    edThetaMin.Text := '0';
    edThetaMax.Text := '360';
    laUnit.Caption := 'DEG';
  end
  else begin
    mOptionsRad.Checked := True;
    laUThetaMin.Visible := False;
    laUThetaMax.Visible := False;
    edThetaMin.Text := '0';
    edThetaMax.Text := '2π';
    laUnit.Caption := 'RAD';
  end;
end;

{ Menu items "Options > Results precision > ...": Select results number of decimal digits }

procedure TfTrigo.mOptionsPrecision0Click(Sender: TObject);

begin
  mOptionsPrecision0.Checked := True;  mOptionsPrecision1.Checked := False;
  mOptionsPrecision2.Checked := False; mOptionsPrecision3.Checked := False;
  iPrecision := 0;
end;

procedure TfTrigo.mOptionsPrecision1Click(Sender: TObject);

begin
  mOptionsPrecision0.Checked := False; mOptionsPrecision1.Checked := True;
  mOptionsPrecision2.Checked := False; mOptionsPrecision3.Checked := False;
  iPrecision := 1;
end;

procedure TfTrigo.mOptionsPrecision2Click(Sender: TObject);

begin
  mOptionsPrecision0.Checked := False; mOptionsPrecision1.Checked := False;
  mOptionsPrecision2.Checked := True;  mOptionsPrecision3.Checked := False;
  iPrecision := 2;
end;

procedure TfTrigo.mOptionsPrecision3Click(Sender: TObject);

begin
  mOptionsPrecision0.Checked := False; mOptionsPrecision1.Checked := False;
  mOptionsPrecision2.Checked := False; mOptionsPrecision3.Checked := True;
  iPrecision := 3;
end;

{ Menu item "Help > Help": Display application help }

procedure TfTrigo.mHelpHelpClick(Sender: TObject);

begin
  if fHelp.Visible then
    fHelp.Close
  else
    fHelp.Show;
end;

{ Menu item "Help > About": Display application about }

procedure TfTrigo.mHelpAboutClick(Sender: TObject);

var
  S: string;

begin
  S := 'Mathematics:' + LineEnding;
  S += 'Solving basic trigonometric equations.' + LineEnding + LineEnding;
  S += 'Version 1.0, © allu, October 2025.';
  MessageDlg('About "Trigonometry2"', S, mtInformation, [mbOK], 0);
end;

{ Button "Solve" pushed: Calculate and display equation solution(s) }

procedure TfTrigo.btSolveClick(Sender: TObject);

const
  Functions: array[0..5] of string = (
    'sin', 'cos', 'tan', 'cotan', 'sec', 'cosec'
  );

var
  FX, I, J: Integer;
  Theta: Real;
  Mess: string;
  NewSolution: Boolean;
  Thetas, ThetasSol: array of Real;

begin
  edPSolution.Text := ''; edISolutions.Text := '';
  // Get equation parameters from the form
  ReadParams(sA, sB, sN, sK, sThetaMin, sThetaMax, rA, rB, rN, rK, rThetaMin, rThetaMax, Mess);
  if Mess = '' then begin
    // Get principal and [0°; 360°] solutions for a·f(θ) = b.
    for I := 0 to 5 do begin
      if sFunction = Functions[I] then
        FX := I;
    end;
    case FX of
      0: GetAngles_EqSin(rA, rB, rTheta0, rTheta1, rTheta2, bHasSolution);
      1: GetAngles_EqCos(rA, rB, rTheta0, rTheta1, rTheta2, bHasSolution);
      2: GetAngles_EqTan(rA, rB, rTheta0, rTheta1, rTheta2, bHasSolution);
      3: GetAngles_EqCotan(rA, rB, rTheta0, rTheta1, rTheta2, bHasSolution);
      4: GetAngles_EqSec(rA, rB, rTheta0, rTheta1, rTheta2, bHasSolution);
      5: GetAngles_EqCosec(rA, rB, rTheta0, rTheta1, rTheta2, bHasSolution);
    end;
    // Equation has no solution
    if not bHasSolution then begin
      MessageDlg('Trigonometric equations', 'This equation has no solution!', mtInformation, [mbOK], 0)
    end
    // Equation has one or more solutions
    else begin
      // Principal solution
      rTheta0 := (rTheta0 - rK) / rN; rTheta0 /= DegRadMult(mOptionsRad.Checked);
      edPSolution.Text := RFormat(rTheta0, iPrecision);
      if not mOptionsRad.Checked then
        edPSolution.Text := edPSolution.Text + '°';
      // Determine interval solutions
      // ----------------------------
      I := 0; SetLength(Thetas, I);
      // Adapt interval limits, considering n and k
      rThetaMin := rThetaMin * rN + rK;
      rThetaMax := rThetaMax * rN + rK;
      // Calculate all possible interval solutions for first angle
      // adding and subtracting 360° until the angle is outside the limits
      Theta := rTheta1;
      repeat
        if (Theta >= rThetaMin) and (Theta <= rThetaMax) then begin
          Inc(I); SetLength(Thetas, I);
          Thetas[I - 1] := Theta;
        end;
        Theta += 2 * Pi;
      until Theta > rThetaMax;
      Theta := rTheta1;
      repeat
        if (Theta >= rThetaMin) and (Theta <= rThetaMax) then begin
          Inc(I); SetLength(Thetas, I);
          Thetas[I - 1] := Theta;
        end;
        Theta -= 2 * Pi;
      until Theta < rThetaMin;
      // Calculate all possible interval solutions for second angle
      // adding and subtracting 360° until the angle is outside the limits
      Theta := rTheta2;
      repeat
        if (Theta >= rThetaMin) and (Theta <= rThetaMax) then begin
          Inc(I); SetLength(Thetas, I);
          Thetas[I - 1] := Theta;
        end;
        Theta += 2 * Pi;
      until Theta > rThetaMax;
      Theta := rTheta2;
      repeat
        if (Theta >= rThetaMin) and (Theta <= rThetaMax) then begin
          Inc(I); SetLength(Thetas, I);
          Thetas[I - 1] := Theta;
        end;
        Theta -= 2 * Pi;
      until Theta < rThetaMin;
      // There is at least one interval solution
      if Length(Thetas) > 0 then begin
        // Sort angles
        for I := 0 to Length(Thetas) - 2 do begin
          for J := I to Length(Thetas) - 1 do begin
            if Thetas[I] > Thetas[J] then begin
              Theta := Thetas[I]; Thetas[I] := Thetas[J]; Thetas[J] := Theta;
            end;
          end;
        end;
        // Eliminate doubles
        Theta := (Thetas[0] - rK) / rN; Theta /= DegRadMult(mOptionsRad.Checked);
        SetLength(ThetasSol, 1); ThetasSol[0] := Theta;
        for I := 2 to Length(Thetas) do begin
          NewSolution := True;
          for J := 1 to I - 1 do begin
            if Thetas[I - 1] = Thetas[J - 1] then
              NewSolution := False;
          end;
          if NewSolution then begin
            Theta := (Thetas[I - 1] - rK) / rN; Theta /= DegRadMult(mOptionsRad.Checked);
            SetLength(ThetasSol, Length(ThetasSol) + 1);
            ThetasSol[Length(ThetasSol) - 1] := Theta;
          end;
        end;
        // Display interval solutions
        for I := 1 to Length(ThetasSol) do begin
          if I > 1 then
            edISolutions.Text := edISolutions.Text + ' ; ';
          edISolutions.Text := edISolutions.Text + RFormat(ThetasSol[I - 1], iPrecision);
          if not mOptionsRad.Checked then
            edISolutions.Text := edISolutions.Text + '°';
        end;
      end
      else begin
        // There is no solution for this interval
        edISolutions.Text := 'No solution';
      end;
    end;
  end
  else begin
    MessageDlg('Invalid data', Mess, mtError, [mbOK], 0);
  end;
end;

{ Button "π" pushed: Insert the "π" symbol in actually selected edit field }

procedure TfTrigo.btPiClick(Sender: TObject);

begin
  if EdEdited <> nil then begin
    edEdited.Text := edEdited.Text + 'π';
    edEdited.SetFocus;
  end;
end;

{ Button "√" pushed: Insert the "√" symbol in actually selected edit field }

procedure TfTrigo.btSquarerootClick(Sender: TObject);

begin
  if EdEdited <> nil then begin
    edEdited.Text := edEdited.Text + '√';
    edEdited.SetFocus;
  end;
end;

{ Function changes (user selected another radio button) }

procedure TfTrigo.rbSinChange(Sender: TObject);

begin
  if rbSin.Checked then
    sFunction := 'sin';
end;

procedure TfTrigo.rbCosChange(Sender: TObject);

begin
  if rbCos.Checked then
    sFunction := 'cos';
end;

procedure TfTrigo.rbTanChange(Sender: TObject);

begin
  if rbTan.Checked then
    sFunction := 'tan';
end;

procedure TfTrigo.rbCotanChange(Sender: TObject);

begin
  if rbCotan.Checked then
    sFunction := 'cotan';
end;

procedure TfTrigo.rbSecChange(Sender: TObject);

begin
  if rbSec.Checked then
    sFunction := 'sec';
end;

procedure TfTrigo.rbCosecChange(Sender: TObject);

begin
  if rbCosec.Checked then
    sFunction := 'cosec';
end;

{ Edit field selection (user clicked in edit filed): Set this field as the one where the "π" and "√" symbols will be inserted }

procedure TfTrigo.edAClick(Sender: TObject);

begin
  edEdited := edA;
end;

procedure TfTrigo.edBClick(Sender: TObject);

begin
  edEdited := edB;
end;

procedure TfTrigo.edNClick(Sender: TObject);

begin
  edEdited := edN;
end;

procedure TfTrigo.edKClick(Sender: TObject);

begin
  edEdited := edK;
end;

procedure TfTrigo.edThetaMinClick(Sender: TObject);

begin
  edEdited := edThetaMin;
end;

procedure TfTrigo.edThetaMaxClick(Sender: TObject);

begin
  edEdited := edThetaMax;
end;

end.

